#Read and Prune Phylogeny
install.packages("RRphylo")
library("RRphylo")
library(geiger)
library(phytools)


phyloPereira <- read.tree('https://ars.els-cdn.com/content/image/1-s2.0-S1055790316304316-mmc2.nwk')
plotTree(phyloPereira, fsize=0.9)
turt<-read.table("Data_turt.txt", header = T, row.names=1)

name.check(phyloPereira,turt)###############

#Prune Tree
species<- turt$Species
cut<-name.check(phyloPereira,turt)
cut2<-as.vector(cut$tree_not_data)
pruned.tree<- drop.tip(phyloPereira, cut2, trim.internal = TRUE, subtree = FALSE,
                       root.edge = 0)

name.check(pruned.tree,turt)



################################ 1. Are reproductive traits correlated as predicted by egg size theory?
clutch.FEC<-as.matrix(turt$clutchmax)
rownames(clutch.FEC) <- rownames(turt)
threesizes<- as.array(cut(clutch.FEC, breaks = c(0, 4,29, Inf), labels = c(1:3)))
rownames(threesizes) <- rownames(turt)


three_size_mapped_tree <- make.simmap(pruned.tree,threesizes, model="ARD", nsim=1)
cols<-setNames(c("blue","coral", "green"),c("1", "2", "3"))
# plot the mapped ancestral states
plot(three_size_mapped_tree, cols, ftype="i", fsize=0.2)
# add a legend
legend("bottomleft", c("small", "medium", "large"), fill=c("blue","coral", "green"))
title(main="Phylogenetic tree mapped with turtles clutch sizes",font.main=3)


#Relative egg size (ESI): egg length/carapace length 
#Egg shape (ESH):egg length/egg width
ESI<- turt$egglenght/turt$carap
ESH<- as.vector(turt$egglenght/turt$eggwidth)
turt.cor<-cbind(ESH, ESI)

class(turt.cor)
rownames(turt.cor) <- rownames(turt)
head(turt.cor)



Q1<-evolvcv.lite(three_size_mapped_tree, turt.cor)

#evolutionary covariance matrices (sometimes called Brownian 'rate matrices') 
#compute evolutionary correlations between the two traits for each one of the regimes

cov2cor(Q1$model1$R)

cov2cor(Q1$model2$R$`1`)
cov2cor(Q1$model2$R$`2`)
cov2cor(Q1$model2$R$`3`)

cov2cor(Q1$model3$R$`1`)
cov2cor(Q1$model3$R$`2`)
cov2cor(Q1$model3$R$`3`)

cov2cor(Q1$model4$R$`1`)
cov2cor(Q1$model4$R$`2`)
cov2cor(Q1$model4$R$`3`)


#Phylomorphospace

phylomorphospace(three_size_mapped_tree,turt.cor,colors=cols,
                 ftype="off",bty="n",lwd=3,node.size=c(0,1.5),
                 node.by.map=TRUE,xlab="Egg size",
                 ylab="Egg shape")
legend("topleft",c("Small clutch","Medium clutch", "Large clutch"),
       pch=21,pt.cex=2,pt.bg=cols,bty="n")
title(main="Phylomorphospace plot of turtles egg strategies",font.main=3)


################################ 2. Do turtles have convergent reproductive strategies?

#Test convergence between cryptodirans and pleurodirans that produce clutches containing bellow 5 eggs, from 5 to 29 eggs, and 30 or more eggs.
turt3<-read.table("3states.txt", header = T, row.names=1)
name.check(pruned.tree,turt3)
turt3states<-as.array(turt3$Character)##################character states for clade convergence test
rownames(turt3states) <- rownames(turt3)
turtTres<-turt3[,-4]
head(turtTres)
SC.state.All<-search.conv(tree=pruned.tree, y=turtTres, state=turt3states,min.dim=1,
                        filename = "3states", declust = TRUE)

# Test divergence among species that produce small and large clutches (containing bellow 5 and above 30 eggs)

turt5<-read.table("Divergent_4.30.txt", header = T, row.names=1)
name.check(pruned.tree,turt5)
Char2<-as.array(turt5$Character)
rownames(Char2) <- rownames(turt5)
turt6<-turt5[,-c(4,5)]
head(turt6)

## search for convergence within species
SC.state.D<-search.conv(tree=pruned.tree, y=turt6, state=Char2,min.dim=1,
                        filename = "Divergent_4.30", declust = TRUE)

#Test convergence between species from tropical and temperate areas###############################################################
#This test was not included in the manuscript
#turtT<-read.table("TxT.txt", header = TRUE, row.names=1)
#name.check(pruned.tree,turtT)
#TT<-as.array(turtT$Character)##################character states for clade convergence test
#rownames(TT) <- rownames(turtT)
#turtTT<-turtT[,-6]
#head(turtTT)
#SC.state.T<-search.conv(tree=pruned.tree, y=turtTT, state=TT,min.dim=1,filename = "Turt_TxT", declust = TRUE)
###################################################################################################################################


############################### 3. Do ecological factors influence egg/clutch characteristics?


############################################################################################################
#########################################################################################################
######################################## PGLS ###########################################################
##########################################################################################################
##########################################################################################################

setwd("~/Desktop/diretorior")

########## Reading Packages ##########
library(GGally)
library(phytools)
library(nlme)
library(vcd)
library(usdm)
library(treeplyr)
library(geiger)
library(rr2)
library(scales)
library(sensiPhy)
library(dplyr)
########## Reading Data ##########

turt<-read.table("turtPereira2_all.csv", header = T, row.names=1, sep=";")

#Egg Size
eggsize <- turt$egglenght/turt$carap #Relative egg size
turt["eggsize"]<- eggsize

#Clutch Mean
clutch_mean <- (turt$clutchmin+turt$clutchmax)/2 #Mean number of eggs per clutch
turt["clutch_mean"]<- clutch_mean

#Fecundity
clutchyear_mean <- ((turt$clutchesyearmax + turt$clutchesyearmin)/2) #Mean number of clutches per year
fecund <- turt$clutchmax*clutchyear_mean  #Maximum number of egg per clutch * Mean number of clutches per year
turt["fecund"]<- fecund

str(turt)

########## Reading Tree ##########

phyloPereira <- read.tree('https://ars.els-cdn.com/content/image/1-s2.0-S1055790316304316-mmc2.nwk')
plotTree(phyloPereira, fsize=0.9)
name.check(phyloPereira,turt) #not ok - more species in the tree than in the dataset

########## Pruned Tree ##########

cut<-name.check(phyloPereira,turt)
cut2<-as.vector(cut$tree_not_data)
pruned.tree<- drop.tip(phyloPereira, cut2, trim.internal = TRUE, subtree = FALSE, root.edge = 0)

name.check(pruned.tree,turt) # ok
plotTree(pruned.tree, fsize=0.9)

########## Plotting categorical data ##########

ggplot(turt, aes(diet, eggsize)) + geom_violin(aes(col=diet, fill=diet)) + theme(legend.position="none") #ok

ggplot(turt, aes(Habitat, eggsize)) + geom_violin(aes(col=Habitat, fill=Habitat)) + theme(legend.position="none") #ok

ggplot(turt, aes(Climatic_zone, eggsize)) + geom_violin(aes(col=Climatic_zone, fill=Climatic_zone)) + theme(legend.position="none") #ok

ggplot(turt, aes(Zoogeography, eggsize)) + geom_violin(aes(col=Climatic_zone, fill=Climatic_zone)) + theme(legend.position="none") #ok

########## Checking for combinations of crossed observations ##########

#Diet and Habitat
ggplot(turt, aes(diet, eggsize)) + geom_violin(aes(col=Habitat, fill=Habitat)) #Carnivore has no terrestrial species

#Diet and Zoogeography
ggplot(turt, aes(diet, eggsize)) + geom_violin(aes(col=Zoogeography, fill=Zoogeography)) #not ok

#Diet and Climatic Zone
ggplot(turt, aes(diet, eggsize)) + geom_violin(aes(col=Climatic_zone, fill=Climatic_zone)) #ok

#Habitat and Climatic Zone
ggplot(turt, aes(Habitat, eggsize)) + geom_violin(aes(col=Climatic_zone, fill=Climatic_zone)) #ok

#Habitat and Zoogeography
ggplot(turt, aes(Habitat, eggsize)) + geom_violin(aes(col=Zoogeography, fill=Zoogeography)) #not ok

#Climatic Zone and Zoogeography
ggplot(turt, aes(Climatic_zone, eggsize)) + geom_violin(aes(col=Zoogeography, fill=Zoogeography)) #not ok

#DROP ZOOGEOGRAPHY
#DO NOT USE HABITAT AND DIET IN THE SAME MODEL

########## Colinearity between pairwise categorical predictors ##########

chisq.test(turt$Climatic_zone, turt$Habitat, correct=F) #not ok
chisq.test(turt$Climatic_zone, turt$diet, correct=F) #not ok
chisq.test(turt$Habitat, turt$diet, correct=F) #ok

#HABITAT= Lack crossed observations with Diet and has collinearity with Climatic_zone 
#DROP HABITAT

########## Checking the distribution of quantitative predictors ##########
#Avoid skwed distributions.

#EGG SIZE PGLS - clutch_mean
ggplot(turt, aes(clutch_mean)) + geom_density(col="red", fill="red") #not ok
ggplot(turt, aes(log(clutch_mean))) + geom_density(col="red", fill="red") #ok
turt["log_clutch_mean"]<- log(turt$clutch_mean)
str(turt)

#FECUNDITY PGLS - eggsize
ggplot(turt, aes(eggsize)) + geom_density(col="red", fill="red") # ok

#PREDICTORS (EGGSIZE): USE CLIMATIC_ZONE, DIET AND LOG_CLUTCH_MEAN
#PREDICTORS (FECUNDITY): USE CLIMATIC_ZONE, DIET AND EGGSIZE

########## PGLS EGGSIZE ##########

pgls_eggsize<- gls(eggsize~Climatic_zone+diet+log_clutch_mean, data=turt, correlation=corPagel(1, pruned.tree), method="ML")

#Checking for model assumptions
plot(pgls_eggsize, abline=c(0,0))
res1<- resid(pgls_eggsize)
hist(res1)
qqnorm(res1)
qqline(res1)
shapiro.test(res1)
#Normality of residuals - ok
#Homoscedasticity ok
#Linearity of the data - ok


#Inference

#Comparing with the null model
pgls_eggsize_null<- gls(eggsize~1, data=turt, correlation=corPagel(1, pruned.tree), method="ML")

anova(pgls_eggsize_null, pgls_eggsize) #Among the set of predictors being in the model, there is at least one having a significant impact on egg size.

anova(pgls_eggsize) #all predictors are significant

summary(pgls_eggsize)
confint(pgls_eggsize, level=0.95) #95% confidence interval


#Visualizing the results
PGLS.egg.figure<-ggplot(turt, aes(log_clutch_mean, eggsize)) + 
  geom_point(aes(col= diet)) + 
  stat_smooth(method="lm") + 
  theme_bw() + 
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.border=element_blank(), axis.line = element_line(colour = "black"), 
        plot.title = element_text(hjust = 0.5), legend.position=c(0.9, 0.85), 
        legend.title = element_text(size=13), legend.background = element_rect(size=0.5, linetype="solid", colour ="black")) + 
  labs(x = "Log Clutch Mean", y= "Egg Size") +
  guides(colour = guide_legend(title.position = "top", title.hjust=0.5, title = "Diet")) +
  ggtitle("Climatic Zone") + 
  facet_wrap(vars(Climatic_zone), scales="free_x")

#saving figure
pdf("PGLS.egg.figure.pdf")
PGLS.egg.figure
dev.off()


########## PGLS FECUNDITY ##########

pgls_fecund<- gls(fecund~Climatic_zone+diet+eggsize, data=turt, correlation=corPagel(1, pruned.tree), method="ML")
plot(pgls_fecund, abline=c(0,0)) #Homoscedasticity - not ok

pgls_fecund<- gls(log(fecund)~Climatic_zone+diet+eggsize, data=turt, correlation=corPagel(1, pruned.tree), method="ML")
plot(pgls_fecund, abline=c(0,0)) #Homoscedasticity - ok

turt["logfecund"]<- log(fecund)

pgls_fecund<- gls(logfecund~Climatic_zone+diet+eggsize, data=turt, correlation=corPagel(1, pruned.tree), method="ML")

#Checking for model assumptions
plot(pgls_fecund, abline=c(0,0))
res2<- resid(pgls_fecund)
hist(res2)
qqnorm(res2)
qqline(res2)
shapiro.test(res2)
#Normality of residuals - ok
#Homoscedasticity ok
#Linearity of the data - ok


#Inference

#Comparing with the null model
pgls_fecund_null<- gls(logfecund ~1, data=turt, correlation=corPagel(1, pruned.tree), method="ML")

anova(pgls_fecund_null, pgls_fecund) #Among the set of predictors being in the model, there is at least one having a significant impact on fecundity.

anova(pgls_fecund) ##Climatic Zone and Eggsize are significant

summary(pgls_fecund)
confint(pgls_fecund, level=0.95) #95% confidence interval

#Visualizing the results
PGLS.fecund.figure<-ggplot(turt, aes(eggsize, logfecund)) + 
  geom_point() + 
  stat_smooth(method="lm") + 
  theme_bw() + 
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.border=element_blank(), axis.line = element_line(colour = "black"), 
        plot.title = element_text(hjust = 0.5)) + 
  labs(x = "Egg Size", y= "Log Fecundity") +
  ggtitle("Climatic Zone") + 
  facet_wrap(vars(Climatic_zone), scales="free_x")


#saving figure
pdf("PGLS.fecund.figure.pdf")
PGLS.fecund.figure
dev.off()


########## SENSITIVITY TEST ##########
library("sensiPhy")

influ <- influ_phylm(eggsize~Climatic_zone+diet+log_clutch_mean, data = turt, phy = pruned.tree)
# To check summary results and most influential species:
summary(influ)
# Visual diagnostics for clade removal:
sensi_plot(influ)
# Most influential species
influ$influential.species


influ2<- influ_phylm(log(fecund)~Climatic_zone+diet+eggsize, data = turt, phy = pruned.tree)
summary(influ2)
influ2$influential.species